/**
* @file: abstract_iface.h
*
* @version: $Id:$
*
* @release: $Name:$
*
* @component: authorization level daemon
*
* @author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* @copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#ifndef ALD_APP_IFACE_APPIFACE_H
#define ALD_APP_IFACE_APPIFACE_H

#include "ald_types.h"

typedef void (*level_change_request_callback_t)(error_code_t result, bool level_changed);

//------------------------------------ functions required by the daemon from the application interface binding --------

/**
 * @brief called to initialize the application interface binding
 * This function is called by the daemon during its initialization. After returning, the application interface binding
 * shall be able to accept incoming requests from applications. The application interface binding might return an
 * error_code. Any code different from RESULT_OK will cause the daemon to cancel the initialization and to exit.
 *
 * @return an errorcode indicating a successful initialization (RESULT_OK) or an error (any other error code).
 */
error_code_t app_iface_init(void);

/**
 * @brief called to deinitialize the application interface binding
 * This function is called by the daemon to shutdown the application interface binding. On a call, the binding must
 * immediately close all connections to application and free all resources, allocated before.
 */
void app_iface_deinit(void);


/**
 * @brief signals the application a beginning security level change
 * This function needs to be implemented by the application interface binding connected with the daemon. Ones called,
 * this bind must broadcast the signal to all applications connected and interested in this signal.
 * @param targeted_level the target security level
 */
void app_iface_signal_level_changing(security_level_t targeted_level);

/**
 * @brief signals the application a changed security level
 * This function needs to be implemented by the application interface binding connected with the daemon. Ones called,
 * this bind must broadcast the signal to all applications connected and interested in this signal.
 * @param new_level the new security level
 */
void app_iface_signal_level_changed(security_level_t new_level);

//------------------------------------ functions provided by the daemon to the application interface binding --------

/**
 * @brief returns the current security level
 * This function is implemented on the daemons side. The application interface binding can use the function to
 * determine the current level and send it to an application on demand. During a level change, this property returns
 * the current level. It is updated ones the change is done before the respective signal is called.
 * @return the current security level
 */
security_level_t app_iface_get_security_level(void);

/**
 * returns the state of the daemon
 * @return the state of the daemon
 */
daemon_level_state_t app_iface_get_daemon_level_state(void);

/**
 * \brief call to notify that the app interface is ready
 */
void app_iface_set_ready(void);

/**
 * \brief call to request the daemon to exit due to errors
 * This functions stops the main loop and exits the daemon with an error code. Any kind of message about
 * the reason must be logged by the application interface binding before.
 *
 * \param error the error code
 */
void app_iface_request_daemon_exit_due_to_errors(error_code_t error);

/**
 * @brief call to request a level change
 * This function signals the daemon a level change request.
 *
 * @param response the response data structure containing the challenge, the targeted level, ...
 * @param callback called after the level change is done
 */
void app_iface_request_level_change(const challenge_response_t *response,
		level_change_request_callback_t callback);

/**
 * @brief call to request locking the device
 * This function signals the daemon a locking request
 * @param callback called after the level change is done
 */
void app_iface_request_lock_device(level_change_request_callback_t callback);

/**
 * @brief call to create a new challenge and return a pointer to it
 * This functions creates a new challenge. The current one is overwritten and hence gets invalid. The caller of this
 * function does not need to worry about memory management. It can call this function as often as wanted.
 *
 * @param ecu_id the ecu_id part of the challenge
 * @return the new challenge
 */
const challenge_t *app_iface_create_new_challenge(const char *ecu_id);

#endif
